package org.hibernate.dialect;
/**
 *  2023/6/19 Support Hibernate 6.2.x
 *  Hibernate 6.2.X
 *  DBMaster 5.4.6 or later
 * 
 */
import java.sql.CallableStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.hibernate.MappingException;
import org.hibernate.boot.Metadata;
import org.hibernate.boot.model.FunctionContributions;
import org.hibernate.boot.model.TypeContributions;
import org.hibernate.boot.model.naming.Identifier;
import org.hibernate.boot.model.relational.QualifiedName;
import org.hibernate.boot.model.relational.QualifiedNameParser;
import org.hibernate.boot.model.relational.SqlStringGenerationContext;
import org.hibernate.boot.spi.MetadataImplementor;
import org.hibernate.dialect.DatabaseVersion;
import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.function.CommonFunctionFactory;
import org.hibernate.dialect.function.StandardSQLFunction;
import org.hibernate.dialect.identity.IdentityColumnSupport;
import org.hibernate.dialect.identity.IdentityColumnSupportImpl;
import org.hibernate.dialect.pagination.AbstractLimitHandler;
import org.hibernate.dialect.pagination.LimitHandler;
import org.hibernate.dialect.pagination.OffsetFetchLimitHandler;
import org.hibernate.dialect.temptable.TemporaryTableKind;
import org.hibernate.engine.jdbc.dialect.spi.DialectResolutionInfo;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.internal.util.StringHelper;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.Constraint;
import org.hibernate.mapping.Table;
import org.hibernate.mapping.UniqueKey;
import org.hibernate.query.spi.Limit;
import org.hibernate.query.spi.QueryEngine;
import org.hibernate.query.sqm.function.SqmFunctionRegistry;
import org.hibernate.query.sqm.mutation.internal.temptable.AfterUseAction;
import org.hibernate.query.sqm.mutation.internal.temptable.BeforeUseAction;
import org.hibernate.service.ServiceRegistry;
import org.hibernate.sql.ast.SqlAstTranslator;
import org.hibernate.sql.ast.SqlAstTranslatorFactory;
import org.hibernate.sql.ast.spi.StandardSqlAstTranslatorFactory;
import org.hibernate.sql.ast.tree.Statement;
import org.hibernate.sql.exec.spi.JdbcOperation;
import org.hibernate.tool.schema.internal.StandardTableExporter;
import org.hibernate.tool.schema.spi.Exporter;
import org.hibernate.type.BasicType;
import org.hibernate.type.BasicTypeRegistry;
import org.hibernate.type.SqlTypes;
import org.hibernate.type.StandardBasicTypes;
import org.hibernate.type.descriptor.sql.internal.DdlTypeImpl;
import org.hibernate.type.descriptor.sql.spi.DdlTypeRegistry;

public class DBMasterDialect extends Dialect {

	public DBMasterDialect() {
		super(DatabaseVersion.make( 5, 4 ));
	}

	public DBMasterDialect(DatabaseVersion version) {
		super(version);
	}

	public DBMasterDialect(DialectResolutionInfo info) {
		super(info);
	}
	
	@Override
	protected String columnType(int sqlTypeCode) {
		switch ( sqlTypeCode ) 
		{
		case SqlTypes.BOOLEAN:
			return "smallint";
		case SqlTypes.TINYINT:
			return "numeric(3,0)"; 
		case SqlTypes.FLOAT:
			return "float";
		case SqlTypes.DOUBLE:
			return "double";
		case SqlTypes.NUMERIC:
			return columnType(SqlTypes.DECIMAL);
		case SqlTypes.DECIMAL:
			return "decimal($p,$s)";
		case SqlTypes.BIT:
			return "binary(1)";
		case SqlTypes.VARBINARY:
			return "BINARY($l)";
		}
		return super.columnType( sqlTypeCode );
	}	

	@Override
	protected void registerColumnTypes(TypeContributions typeContributions, ServiceRegistry serviceRegistry) {
		super.registerColumnTypes( typeContributions, serviceRegistry );
		final DdlTypeRegistry ddlTypeRegistry = typeContributions.getTypeConfiguration().getDdlTypeRegistry();

		ddlTypeRegistry.addDescriptor( new DdlTypeImpl( SqlTypes.JSON, "jsoncols", this ) );
	}	
	
	@Override
	public void initializeFunctionRegistry(FunctionContributions functionContributions) {
		super.initializeFunctionRegistry(functionContributions);

			CommonFunctionFactory functionFactory = new CommonFunctionFactory(functionContributions);
		
		functionFactory.ascii();
		functionFactory.cosh();
		functionFactory.cot();
		functionFactory.characterLength_len();
		functionFactory.nowCurdateCurtime();
		functionFactory.datepartDatename();
		functionFactory.daynameMonthname();
		functionFactory.dayofweekmonthyear();
		functionFactory.daysBetween();
		functionFactory.degrees();
		functionFactory.insert();
		functionFactory.lastDay();
		functionFactory.locate();
		functionFactory.log();
		functionFactory.log10();
		functionFactory.trim2();
		functionFactory.hourMinuteSecond();
		functionFactory.pi();
		functionFactory.weekQuarter();
		functionFactory.radians();
		functionFactory.rand();
		functionFactory.repeat();
		functionFactory.replace();
		functionFactory.sinh();
		functionFactory.space();
		functionFactory.tanh();
		functionFactory.yearMonthDay();
		
		BasicTypeRegistry basicTypeRegistry = functionContributions.getTypeConfiguration().getBasicTypeRegistry();		
		SqmFunctionRegistry functionRegistry = functionContributions.getFunctionRegistry();

		BasicType<String> stringType = basicTypeRegistry.resolve(StandardBasicTypes.STRING);
		BasicType<Integer> integerType = basicTypeRegistry.resolve(StandardBasicTypes.INTEGER);
		BasicType<Date> dateType = basicTypeRegistry.resolve(StandardBasicTypes.DATE);
		
		functionRegistry.register( "add_days", new StandardSQLFunction("add_days", StandardBasicTypes.DATE) );
		functionRegistry.register( "add_hours", new StandardSQLFunction("add_hours", StandardBasicTypes.TIME) );
		functionRegistry.register( "add_mins", new StandardSQLFunction("add_mins", StandardBasicTypes.TIME) );
		functionRegistry.register( "add_months", new StandardSQLFunction("add_months", StandardBasicTypes.DATE) );		
		functionRegistry.register( "add_secs", new StandardSQLFunction("add_secs", StandardBasicTypes.TIME) );
		
		functionRegistry.register( "atan2", new StandardSQLFunction("atan2", StandardBasicTypes.DOUBLE) );		
		functionRegistry.register( "atof", new StandardSQLFunction("atof", StandardBasicTypes.DOUBLE) );		

		functionRegistry.register( "bloblen", new StandardSQLFunction("bloblen", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "char", new StandardSQLFunction("char", StandardBasicTypes.STRING) );		
		functionRegistry.registerAlternateKey( "char_length", "character_length" );
		
		functionRegistry.noArgsBuilder("current_user")
			.setInvariantType(stringType)
			.setUseParenthesesWhenNoArgs( true )
			.register();
		functionRegistry.noArgsBuilder("database")
			.setInvariantType(stringType)
			.setUseParenthesesWhenNoArgs( true )
			.register();
		
		functionRegistry.register( "dmlic", new StandardSQLFunction("dmlic", StandardBasicTypes.STRING) );
		
		functionRegistry.register( "fileexist", new StandardSQLFunction("fileexist", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "filelen", new StandardSQLFunction("filelen", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "filename", new StandardSQLFunction("filename", StandardBasicTypes.STRING) );
                                  
		functionRegistry.register( "fix", new StandardSQLFunction("fix", StandardBasicTypes.INTEGER) );		
		functionRegistry.register( "frexpe", new StandardSQLFunction("frexpe", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "frexpm", new StandardSQLFunction("frexpm", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "ftoa", new StandardSQLFunction("ftoa", StandardBasicTypes.STRING) );	
		functionRegistry.register( "highlight", new StandardSQLFunction("highlight", StandardBasicTypes.BLOB) );	
		functionRegistry.register( "hitcount", new StandardSQLFunction("hitcount", StandardBasicTypes.INTEGER) );	
		functionRegistry.register( "hitpos", new StandardSQLFunction("hitpos", StandardBasicTypes.INTEGER) );
		                  
		functionRegistry.register( "hms", new StandardSQLFunction("hms", StandardBasicTypes.TIME) );
		functionRegistry.register( "htmlhighlight", new StandardSQLFunction("htmlhighlight", StandardBasicTypes.BLOB) );
		//functionRegistry.register("htmltitle", new StandardSQLFunction("htmltitle", StandardBasicTypes.VARCHAR) );
		functionRegistry.register( "hypot", new StandardSQLFunction("hypot", StandardBasicTypes.DOUBLE) );

		functionRegistry.register( "invdate", new StandardSQLFunction("invdate", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "invtime", new StandardSQLFunction("invtime", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "invtimestamp", new StandardSQLFunction("invtimestamp", StandardBasicTypes.INTEGER) );

		functionRegistry.register( "lcase", new StandardSQLFunction("lcase", StandardBasicTypes.STRING) );
		functionRegistry.register( "ldexp", new StandardSQLFunction("ldexp", StandardBasicTypes.DOUBLE) );

		functionRegistry.register( "mdy", new StandardSQLFunction("mdy", StandardBasicTypes.DATE) );
		functionRegistry.register( "modfi", new StandardSQLFunction("modfi", StandardBasicTypes.DOUBLE) );
		functionRegistry.register( "modfm", new StandardSQLFunction("modfm", StandardBasicTypes.DOUBLE) );
		functionRegistry.register( "month", new StandardSQLFunction("month", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "next_day", new StandardSQLFunction("next_day", StandardBasicTypes.DATE) );
		
		//functionRegistry.register("pi", new NoArgSQLFunction("pi", StandardBasicTypes.DECIMAL) );
		functionRegistry.register( "pow", new StandardSQLFunction("pow", StandardBasicTypes.DOUBLE) );

		functionRegistry.noArgsBuilder("rnd")
			.setInvariantType(integerType)
			.setUseParenthesesWhenNoArgs( true )
			.register();
		
		functionRegistry.register( "secs_between", new StandardSQLFunction("secs_between", StandardBasicTypes.INTEGER) );

		functionRegistry.noArgsBuilder("session_user")
		.setInvariantType(stringType)
		.setUseParenthesesWhenNoArgs( false )
		.register();
		
		functionRegistry.register( "strtoint", new StandardSQLFunction("strtoint", StandardBasicTypes.INTEGER) );
		functionRegistry.register( "subblob", new StandardSQLFunction("subblob", StandardBasicTypes.BLOB) );
		functionRegistry.register( "subblobtobin", new StandardSQLFunction("subblobtobin", StandardBasicTypes.BINARY) );
		functionRegistry.register( "subblobtochar", new StandardSQLFunction("subblobtochar", StandardBasicTypes.STRING) );
		
		functionRegistry.register( "timepart", new StandardSQLFunction("timepart", StandardBasicTypes.DATE) );
		functionRegistry.register( "timestampadd", new StandardSQLFunction("timestampadd", StandardBasicTypes.TIMESTAMP) );
		functionRegistry.register( "timestampdiff", new StandardSQLFunction("timestampdiff", StandardBasicTypes.DOUBLE) );
		
		functionRegistry.namedDescriptorBuilder( "to_date" )
			.setParameterTypes( org.hibernate.query.sqm.produce.function.FunctionParameterType.STRING)
			.setInvariantType(dateType)
			.register();		
		//functionRegistry.register("trim", new StandardSQLFunction("trim") );
		functionRegistry.register( "ucase", new StandardSQLFunction("ucase", StandardBasicTypes.STRING ) );

		functionRegistry.noArgsBuilder("user")
			.setInvariantType(stringType)
			.setUseParenthesesWhenNoArgs( false )
			.register();
	}
	
	/*
	 * @return true if the correct order is limit, offset
	 */
	@Override
	public String getAddColumnString() {
		return "add column";
	}	

	@Override
	public String getCurrentTimestampSelectString() {
		return "select current_timestamp";
	}
	
	@Override
	public String getDropForeignKeyString() {
		return " drop foreign key ";
	}	

	private DBMasterLimitHandler limitHandler = new DBMasterLimitHandler(true);

	public class DBMasterLimitHandler extends OffsetFetchLimitHandler 
	{
		public DBMasterLimitHandler(boolean variableLimit) {
			super(variableLimit);
		}

		@Override
		public String processSql(String sql, Limit limit)
		{
			boolean hasFirstRow = hasFirstRow(limit);
			boolean hasMaxRows = hasMaxRows(limit);

			if ( !hasFirstRow && !hasMaxRows ) {
				return sql;
			}

			StringBuilder offsetFetch = new StringBuilder();

			offsetFetch.append( " limit " );				
			if ( supportsVariableLimit() ) {
				offsetFetch.append( "?" );
			}
			else {
				offsetFetch.append( getMaxOrLimit( limit ) );
			}
			if ( hasFirstRow ) {
				offsetFetch.append( " , " );
				if ( supportsVariableLimit() ) {
					offsetFetch.append( "?" );
				}
				else {
					offsetFetch.append( limit.getFirstRow() );
				}
			}

			return insertBeforeForUpdate( offsetFetch.toString(), sql );
		}
		
	}
	
	@Override
	public LimitHandler getLimitHandler() 
	{
		return limitHandler;
	}
		
	@Override
	public ResultSet getResultSet(CallableStatement ps) throws SQLException {
		boolean isResultSet = ps.execute();
		while (!isResultSet && ps.getUpdateCount() != -1) {
			isResultSet = ps.getMoreResults();
		}
		return ps.getResultSet();
	}
	
	@Override
	public boolean isCurrentTimestampSelectStringCallable() {
		return false;
	}	
	
	@Override
	public boolean qualifyIndexName() {
		return false;
	}
	
	@Override
	public int registerResultSetOutParameter(CallableStatement statement, int col) throws SQLException {
		return col;
	}
	
	@Override
	public boolean supportsCurrentTimestampSelection() {
		return true;
	}
	
	@Override
	public boolean supportsTemporaryTables() {
		return true;
	}
		
	@Override
	public TemporaryTableKind getSupportedTemporaryTableKind() {
		return TemporaryTableKind.LOCAL;
	}

	@Override
	public String getTemporaryTableCreateOptions() {
		return "";
	}

	@Override
	public String getTemporaryTableCreateCommand() {
		return "create temp table";
	}

	@Override
	public AfterUseAction getTemporaryTableAfterUseAction() {
		return AfterUseAction.NONE;
	}

	@Override
	public BeforeUseAction getTemporaryTableBeforeUseAction() {
		return BeforeUseAction.CREATE;
	}

	// add for drop table if exists
	@Override
	public boolean supportsIfExistsAfterTableName() {
		return false;
	}
	
	@Override
	public boolean supportsLobValueChangePropagation() {
		return false;
	}
	
	@Override
	public boolean supportsTupleDistinctCounts() {
		return false;
	}

	@Override
	public boolean doesReadCommittedCauseWritersToBlockReaders() {
		return true;
	}

	@Override
	public boolean supportsIfExistsBeforeTableName() {
		return true;
	}
	
	@Override
	public boolean supportsLockTimeouts() {
		return true;
	}
	
	@Override
	public boolean supportsExistsInSelect(){
		return true;
	} 
	
	@Override
	public boolean supportsCascadeDelete() {
		return true;
	}

	@Override
	public String getCascadeConstraintsString() {
		return " cascade";
	}

	private DBMasterTableExporter tableExporter = new DBMasterTableExporter(this);
	@Override
	public Exporter<Table> getTableExporter() {
		return tableExporter;
	}
	
	public class DBMasterTableExporter extends StandardTableExporter 
	{

		public DBMasterTableExporter(Dialect dialect) {
			super(dialect);
			// TODO Auto-generated constructor stub
		}

		@Override
		public String[] getSqlCreateStrings(
				Table table,
				Metadata metadata,
				SqlStringGenerationContext context) {
			final QualifiedName tableName = new QualifiedNameParser.NameParts(
					Identifier.toIdentifier( table.getCatalog(), table.isCatalogQuoted() ),
					Identifier.toIdentifier( table.getSchema(), table.isSchemaQuoted() ),
					table.getNameIdentifier()
			);

			try {
				String formattedTableName = context.format( tableName );
				StringBuilder buf =
						new StringBuilder( tableCreateString( table.hasPrimaryKey() ) )
								.append( ' ' )
								.append( formattedTableName )
								.append( " (" );


				boolean isPrimaryKeyIdentity = table.hasPrimaryKey()
						&& table.getIdentifierValue() != null
						&& table.getIdentifierValue().isIdentityColumn( ( (MetadataImplementor) metadata ).getMetadataBuildingOptions().getIdentifierGeneratorFactory(), dialect );
				// this is the much better form moving forward as we move to metamodel
				//boolean isPrimaryKeyIdentity = hasPrimaryKey
				//				&& table.getPrimaryKey().getColumnSpan() == 1
				//				&& table.getPrimaryKey().getColumn( 0 ).isIdentity();

				// Try to find out the name of the primary key in case the dialect needs it to create an identity
				String pkColName = null;
				if ( table.hasPrimaryKey() ) {
					Column pkColumn = table.getPrimaryKey().getColumns().iterator().next();
					pkColName = pkColumn.getQuotedName( dialect );
				}

				boolean isFirst = true;
				for ( Column col : table.getColumns() ) {
					if ( isFirst ) {
						isFirst = false;
					}
					else {
						buf.append( ", " );
					}

					String colName = col.getQuotedName( dialect );
					buf.append( colName );

					if ( isPrimaryKeyIdentity && colName.equals( pkColName ) ) {
						// to support dialects that have their own identity data type
						if ( dialect.getIdentityColumnSupport().hasDataTypeInIdentityColumn() ) {
							buf.append( ' ' ).append(
									col.getSqlType( metadata.getDatabase().getTypeConfiguration(), dialect, metadata )
							);
						}
						String identityColumnString = dialect.getIdentityColumnSupport()
								.getIdentityColumnString( col.getSqlTypeCode(metadata) );
						buf.append( ' ' ).append( identityColumnString );
					}
					else {
						final String columnType = col.getSqlType(
								metadata.getDatabase().getTypeConfiguration(),
								dialect,
								metadata
						);
						if ( col.getGeneratedAs()==null || dialect.hasDataTypeBeforeGeneratedAs() ) {
							buf.append( ' ' ).append( columnType );
						}

						String defaultValue = col.getDefaultValue();
						if ( defaultValue != null ) {
							buf.append( " default " ).append( defaultValue );
						}

						String generatedAs = col.getGeneratedAs();
						if ( generatedAs != null) {
							buf.append( dialect.generatedAs( generatedAs ) );
						}

						if ( col.isNullable() ) {
							buf.append( dialect.getNullColumnString( columnType ) );
						}
						else if( columnType.compareToIgnoreCase("serial") != 0 && columnType.compareToIgnoreCase("bigserial") !=0 ){ // DBMaster serial do not append not null
							buf.append( " not null" );
						}

					}

					if ( col.isUnique() ) {
						String keyName = Constraint.generateName( "UK_", table, col );
						UniqueKey uk = table.getOrCreateUniqueKey( keyName );
						uk.addColumn( col );
						buf.append(
								dialect.getUniqueDelegate()
										.getColumnDefinitionUniquenessFragment( col, context )
						);
					}

					String checkConstraint = col.checkConstraint();
					if ( checkConstraint != null && dialect.supportsColumnCheck() ) {
						buf.append( checkConstraint );
					}

					String columnComment = col.getComment();
					if ( columnComment != null ) {
						buf.append( dialect.getColumnComment( columnComment ) );
					}
				}
				if ( table.hasPrimaryKey() ) {
					buf.append( ", " )
							.append( table.getPrimaryKey().sqlConstraintString( dialect ) );
				}

				buf.append( dialect.getUniqueDelegate().getTableCreationUniqueConstraintsFragment( table, context ) );

				applyTableCheck( table, buf );

				buf.append( ')' );

				if ( table.getComment() != null ) {
					buf.append( dialect.getTableComment( table.getComment() ) );
				}

				applyTableTypeString( buf );

				List<String> sqlStrings = new ArrayList<>();
				sqlStrings.add( buf.toString() );

				applyComments( table, formattedTableName, sqlStrings );

				applyInitCommands( table, sqlStrings, context );

				return sqlStrings.toArray(StringHelper.EMPTY_STRINGS);
			}
			catch (Exception e) {
				throw new MappingException( "Error creating SQL create commands for table : " + tableName, e );
			}
		}

	}
	
	public IdentityColumnSupport getIdentityColumnSupport(){
		return new DBMasterIdentityColumnSupport();
	}
	
	public class DBMasterIdentityColumnSupport extends IdentityColumnSupportImpl {
		@Override
		public boolean supportsIdentityColumns() {
			return true;
		}

		@Override
		public String getIdentitySelectString(String table, String column, int type) {
			return "select LAST_SERIAL FROM SYSCONINFO";
		}

		@Override
		public String getIdentityColumnString(int type) {
			//starts with 1, implicitly
			return "";
		}
	}	

	@Override
	public SqlAstTranslatorFactory getSqlAstTranslatorFactory() {
		return new StandardSqlAstTranslatorFactory() {
			@Override
			protected <T extends JdbcOperation> SqlAstTranslator<T> buildTranslator(
					SessionFactoryImplementor sessionFactory, Statement statement) {
				return new MySQLSqlAstTranslator<>( sessionFactory, statement );
			}
		};
	}	
}
